/* Any copyright is dedicated to the Public Domain.
   http://creativecommons.org/publicdomain/zero/1.0/ */

"use strict";

/**
 * Check that the runtime info is correctly displayed for ThisIceCat.
 * Also acts as basic sanity check for the default mock of the this-icecat client.
 */

add_task(async function () {
  // Setup a mock for our runtime client factory to return the default THIS_ICECAT client
  // when the client for the this-icecat runtime is requested.
  const runtimeClientFactoryMock = createRuntimeClientFactoryMock();
  const thisIceCatClient = createThisIceCatClientMock();
  runtimeClientFactoryMock.createClientForRuntime = runtime => {
    const {
      RUNTIMES,
    } = require("resource://devtools/client/aboutdebugging/src/constants.js");
    if (runtime.id === RUNTIMES.THIS_ICECAT) {
      return thisIceCatClient;
    }
    throw new Error("Unexpected runtime id " + runtime.id);
  };

  info("Enable mocks");
  enableRuntimeClientFactoryMock(runtimeClientFactoryMock);
  registerCleanupFunction(() => {
    disableRuntimeClientFactoryMock();
  });

  const { document, tab, window } = await openAboutDebugging();
  await selectThisIceCatPage(document, window.AboutDebugging.store);

  info("Check that the 'This IceCat' mock is properly displayed");
  const thisIceCatRuntimeInfo = document.querySelector(".qa-runtime-name");
  ok(
    thisIceCatRuntimeInfo,
    "Runtime info for this-icecat runtime is displayed"
  );
  const runtimeInfoText = thisIceCatRuntimeInfo.textContent;
  ok(
    runtimeInfoText.includes("IceCat"),
    "this-icecat runtime info shows the correct runtime name: " +
      runtimeInfoText
  );
  ok(
    runtimeInfoText.includes("63.0"),
    "this-icecat runtime info shows the correct version number: " +
      runtimeInfoText
  );

  await removeTab(tab);
});
